//
//  MCMirrorSyncDaemon.h
//  MCClientServer
//
//  Created by Brent Gulanowski on 10-02-09.
//  Copyright 2010 Marketcircle Inc. All rights reserved.
//


#import <MCClientServer/MCDatabaseSyncDaemon.h>

#import <MCClientServer/MCClientServerDefines.h>


#define MirrorDaemon ((MCMirrorSyncDaemon *)[MCPGDaemon daemon])


extern NSString *MCMirrorLogFileName;


/* Support Launch Options
 * No arguments:
 *  -a  sync automatically (further arguments ignored)
 * With arguments:
 *  -s  "object" (if archived) or "plist" (if a property list); reads the DBMS access info (host, port, dbname, username, password) from stdin
 *       (for more info, see MCDaemon.h)
 *  -h  hostname
 *  -n  port number
 *  -d  database name
 *  -u  username
 *  -p  password
 *  -c  cloud account
 */

@class MCReplicaClient;
@class MCPDatabaseConnection, MCPConnectionInfo, MCPModel;

@interface MCMirrorSyncDaemon : MCDatabaseSyncDaemon {

	MCReplicaClient *replicaClient;
		
	int exitCode;
	BOOL goingToSleep;
	BOOL recoverOnReset;
	BOOL uninstall;
	BOOL onDemand;
	
	// run options
	NSDictionary *connectionDetails;
	BOOL syncAll;
	
	NSDistributedLock *lock;
}

@property (retain) MCReplicaClient *replicaClient;
@property (readonly) int exitCode;
@property (readwrite) BOOL recoverOnReset;
@property (readwrite) BOOL uninstall; // will uninstall if set
@property (readwrite) BOOL onDemand; // will uninstall if set and no databases exist; defaults to YES

@property (retain) NSDictionary *connectionDetails;
@property (readwrite) BOOL syncAll;

+ (NSDictionary *)authenticationInfoFromOptions:(NSDictionary *)options;

- (void)configureClientTrackingWithUUID:(NSString *)dbUUID;

/*!
 Give subclassers the opportunity to instantiate a subclass of MCReplicaClient. The default implementation simply creates an autoreleased instance of MCReplicaClient. A replica client subclass can perform special app specific activities such as conflict resolution.
 */
- (MCReplicaClient *)instantiateReplicaClient;

- (MCPDatabaseConnection *)clientConnectionForReplica:(NSString *)replicaName;
- (MCPDatabaseConnection *)serverConnectionWithInfo:(MCPConnectionInfo *)info replicaUUID:(NSString *)replicaUUID;

- (MCReplicaClient *)replicaClientWithDetails:(NSDictionary *)details database:(NSString *)dbName;
- (int)syncWithAuthenticationDetails:(NSDictionary *)details;
- (int)syncWithRemoteConnectionDetails:(NSDictionary *)details;

- (int)syncAllReplicas;
- (void)didCreateReplicaForClient:(MCReplicaClient *)client;

- (BOOL)updateAuthenticationStatus:(MCSAuthenticationErrorCode)errorCode forReplicaClient:(MCReplicaClient *)client;

 // endpoint for reaching access daemon
- (MCPGServer *)clientServer;
- (BOOL)prepareSubstrate;
- (void)pruneReplicaAccessInfo;
- (void)recoverLostUsernames;
- (void)scheduleHangDetection;
- (NSString *)accessdServiceIdentifier;

/*!
 Removes the database from the local server. Leaves other stuff alone. Subclassers can clean up further.
 */
- (void)wipeDatabaseForReplicaClient:(MCReplicaClient *)client;

/*!
 Subclassers can do stuff like upload files. At this point, you've been authenticated and have both the local and remote connection. The default implementation does nothing.
 */
- (void)willSyncWithReplicaClient:(MCReplicaClient *)client;

/*!
 Subclassers can do stuff like notify the server of the number changes processes. The default implementation does nothing.
 */
- (void)didSyncWithReplicaClient:(MCReplicaClient *)client;

/*!
 Subclassers can decide what to do in this case. Check the exit code.
 */
- (void)didEncounterExitError:(MCReplicaClient *)client;
@end
